<?php
namespace AIOSEO\Plugin\Common\Social;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles the Twitter meta.
 *
 * @since 4.0.0
 */
class Twitter {

	/**
	 * Returns the Twitter URL for the site.
	 *
	 * @since 4.0.0
	 *
	 * @return string The Twitter URL.
	 */
	public function getTwitterUrl() {
		if ( ! aioseo()->options->social->profiles->sameUsername->enable ) {
			return aioseo()->options->social->profiles->urls->twitterUrl;
		}

		$userName = aioseo()->options->social->profiles->sameUsername->username;
		return ( $userName && in_array( 'twitterUrl', aioseo()->options->social->profiles->sameUsername->included, true ) )
			? 'https://twitter.com/' . $userName
			: '';
	}

	/**
	 * Returns the Twitter card type.
	 *
	 * @since 4.0.0
	 *
	 * @return string $card The card type.
	 */
	public function getCardType() {
		if ( is_home() && 'posts' === get_option( 'show_on_front' ) ) {
			return aioseo()->options->social->twitter->homePage->cardType;
		}

		$metaData = aioseo()->meta->metaData->getMetaData();
		return ! empty( $metaData->twitter_card ) && 'default' !== $metaData->twitter_card ? $metaData->twitter_card : aioseo()->options->social->twitter->general->defaultCardType;
	}

	/**
	 * Returns the Twitter creator.
	 *
	 * @since 4.0.0
	 *
	 * @return string The creator.
	 */
	public function getCreator() {
		$author = '';
		$post   = aioseo()->helpers->getPost();
		if ( $post && aioseo()->options->social->twitter->general->showAuthor ) {
			$twitterUser = get_the_author_meta( 'aioseo_twitter', $post->post_author );
			$author      = aioseo()->social->twitter->prepareUsername( $twitterUser );
		}
		return $author;
	}

	/**
	 * Returns the Twitter image URL.
	 *
	 * @since 4.0.0
	 *
	 * @return string The image URL.
	 */
	public function getImage() {
		$post = aioseo()->helpers->getPost();
		if ( is_home() && 'posts' === get_option( 'show_on_front' ) ) {
			$image = aioseo()->options->social->twitter->homePage->image;
			if ( empty( $image ) ) {
				$image = aioseo()->options->social->facebook->homePage->image;
			}
			if ( empty( $image ) ) {
				$image = aioseo()->social->image->getImage( 'twitter', aioseo()->options->social->twitter->general->defaultImageSourcePosts, $post );
			}

			return $image ? $image : aioseo()->social->facebook->getImage();
		}

		$metaData = aioseo()->meta->metaData->getMetaData( $post );

		if ( ! empty( $metaData->twitter_use_og ) ) {
			return aioseo()->social->facebook->getImage();
		}

		$image = '';
		if ( ! empty( $metaData ) ) {
			$imageSource = ! empty( $metaData->twitter_image_type ) && 'default' !== $metaData->twitter_image_type
				? $metaData->twitter_image_type
				: aioseo()->options->social->twitter->general->defaultImageSourcePosts;

			$image = aioseo()->social->image->getImage( 'twitter', $imageSource, $post );
		}

		return $image ? $image : aioseo()->social->facebook->getImage();
	}

	/**
	 * Returns the Twitter title for the current page.
	 *
	 * @since 4.0.0
	 *
	 * @param  WP_Post|integer $post The post object or ID (optional).
	 * @return string                The Twitter title.
	 */
	public function getTitle( $post = null ) {
		if ( is_home() && 'posts' === get_option( 'show_on_front' ) ) {
			$title = aioseo()->meta->title->prepareTitle( aioseo()->options->social->twitter->homePage->title );
			return $title ? $title : aioseo()->social->facebook->getTitle( $post );
		}

		$post     = aioseo()->helpers->getPost( $post );
		$metaData = aioseo()->meta->metaData->getMetaData( $post );

		if ( ! empty( $metaData->twitter_use_og ) ) {
			return aioseo()->social->facebook->getTitle( $post );
		}

		$title = '';
		if ( ! empty( $metaData->twitter_title ) ) {
			$title = aioseo()->meta->title->prepareTitle( $metaData->twitter_title );
		}

		return $title ? $title : aioseo()->social->facebook->getTitle( $post );
	}

	/**
	 * Returns the Twitter description for the current page.
	 *
	 * @since 4.0.0
	 *
	 * @param  WP_Post|integer $post The post object or ID (optional).
	 * @return string                The Twitter description.
	 */
	public function getDescription( $post = null ) {
		if ( is_home() && 'posts' === get_option( 'show_on_front' ) ) {
			$description = aioseo()->meta->description->prepareDescription( aioseo()->options->social->twitter->homePage->description );
			return $description ? $description : aioseo()->social->facebook->getDescription( $post );
		}

		$post     = aioseo()->helpers->getPost( $post );
		$metaData = aioseo()->meta->metaData->getMetaData( $post );

		if ( ! empty( $metaData->twitter_use_og ) ) {
			return aioseo()->social->facebook->getDescription( $post );
		}

		$description = '';
		if ( ! empty( $metaData->twitter_description ) ) {
			$description = aioseo()->meta->description->prepareDescription( $metaData->twitter_description );
		}

		return $description ? $description : aioseo()->social->facebook->getDescription( $post );
	}

	/**
	 * Prepare twitter username for public display.
	 *
	 * We do things like strip out the URL, etc and return just (at)username.
	 * At the moment, we'll check for 1 of 3 things... (at)username, username, and https://twitter.com/username.
	 *
	 * @since 4.0.0
	 *
	 * @param  string  $profile   Twitter username.
	 * @param  boolean $includeAt Whether or not ot include the @ sign.
	 * @return string             Full Twitter username.
	 */
	public function prepareUsername( $profile, $includeAt = true ) {
		if ( preg_match( '/^(\@)?[A-Za-z0-9_]+$/', $profile ) ) {
			if ( '@' !== $profile[0] && $includeAt ) {
				$profile = '@' . $profile;
			} elseif ( '@' === $profile[0] && ! $includeAt ) {
				$profile = ltrim( $profile, '@' );
			}
		} elseif ( strpos( $profile, 'twitter.com' ) ) {
			$profile = esc_url( $profile );

			// extract the twitter username from the url.
			$parsedTwitterProfile = wp_parse_url( $profile );

			$path      = $parsedTwitterProfile['path'];
			$pathParts = explode( '/', $path );
			$profile   = $pathParts[1];

			if ( $profile ) {
				if ( '@' !== $profile[0] && $includeAt ) {
					$profile = '@' . $profile;
				} elseif ( '@' === $profile[0] && ! $includeAt ) {
					$profile = ltrim( $profile, '@' );
				}
			}
		}

		return $profile;
	}

	/**
	 * Get additional twitter data.
	 *
	 * @since 4.0.0
	 *
	 * @return array An array of additional twitter data.
	 */
	public function getAdditionalData() {
		if ( ! aioseo()->options->social->twitter->general->additionalData ) {
			return [];
		}

		$data = [];
		$post = aioseo()->helpers->getPost();

		if ( $post->post_author ) {
			$data[] = [
				'label' => __( 'Written by', 'all-in-one-seo-pack' ),
				'value' => get_the_author_meta( 'display_name', $post->post_author )
			];
		}

		if ( ! empty( $post->post_content ) ) {
			$minutes = aioseo()->helpers->getReadingTime( $post->post_content );
			if ( ! empty( $minutes ) ) {
				$data[] = [
					'label' => __( 'Est. reading time', 'all-in-one-seo-pack' ),
					// Translators: 1 - The estimated reading time.
					'value' => sprintf( _n( '%1$s minute', '%1$s minutes', $minutes, 'all-in-one-seo-pack' ), $minutes )
				];
			}
		}

		return $data;
	}
}